# -*- coding: utf-8 -*-
#################################################################################
#
#   Copyright (c) 2015-Present Webkul Software Pvt. Ltd. (<https://webkul.com/>)
#   See LICENSE file for full copyright and licensing details.
#   If not, see <https://store.webkul.com/license.html/>
#
#################################################################################

from odoo import api, fields, models, _
import logging
from odoo.exceptions import ValidationError
from odoo.addons.seo_url_redirect.models.ir_http import slug as http_slug
from odoo.http import request

_logger = logging.getLogger(__name__)
VAL_DICT = {'product': 'product_template',
            'category': 'product_public_category'}


class WebsiteSeoRewriteConfig(models.Model):
    _name = 'seo.rewrite.config'
    _description = "seo rewrite configuration"
    _inherit = ['mail.thread', 'mail.activity.mixin']

    active = fields.Boolean('Active', default=False, tracking=True,)
    name = fields.Char("Name", default="Seo Rewrite Configuration")
    state = fields.Selection([
        ('draft', 'Draft'),
        ('ready', 'Ready'),
        ('disabled', 'Disabled'),
        ('enabled', 'Locked')
    ], 'Status', default='draft', required=True, readonly=True, tracking=True,)
    use_suffix = fields.Boolean(string="Use Suffix in URL", tracking=True,)
    seo_model = fields.Selection(
        [('product', 'Product'), ('category', 'Category')])
    suffix_value = fields.Char(
        string="Suffix Value", tracking=True, default='.html')
    pattern_url_key = fields.Char(string="Pattern For Url Key", tracking=True,
                                  help="You can provide field name like(eg: id,name).")
    use_category_url = fields.Boolean(string="Use Category URL on Product",
                                      help="""Enable to append category along with product URL in Website""", tracking=True,)
    use_category_hierarchy = fields.Boolean(
        string="Use Category hierarchy", help="""Enable to manage category hierarchy option in Website""", tracking=True,)
    use_server_rewrites = fields.Boolean(
        string="Use Web Server Rewrites",
        help="""
            By enabling this feature page key will remove from url
            For Example :
            '/shop/catalog-product' => '/catalog-product'
            '/shop/category/category-url' => '/category-url'
            """, tracking=True,

    )
    has_seo_gen = fields.Boolean(
        "Has seo url keys generated for this configuration?")
    activate_cron = fields.Boolean("Run Cron For Redirects", default=False)
    seo_cron = fields.Many2one('ir.cron', string='Seo Cron')
    next_exec_call = fields.Datetime(
        "Next Execution Date", related="seo_cron.nextcall")
    remove_categ_params = fields.Boolean(string="Remove Category params",
                                         help="""Enabling this option will remove the category as an extra query parameter from the URL.""")
    remove_attr_params = fields.Boolean(string="Remove Attribute Parameters",
                                        help="""Enabling this option will remove hash attribute values from the product URL.""")

    @api.onchange('activate_cron')
    def _onchange_activate_cron(self):
        if self.activate_cron and self.active:
            self.seo_cron.active = True
        else:
            self.seo_cron.active = False

    def toggle_active(self):
        res = super(WebsiteSeoRewriteConfig, self).toggle_active()
        if not (self.active):
            self.seo_cron.active = False
        elif self.activate_cron:
            self.seo_cron.active = True


    def reset_to_draft_warning(self):
        msg = """
        <h2 class="text-danger">Need Attention</h2>
            <ul>
                <strong><li>After confirmation,
                all active website redirects and url key related to this configuration will be inactive in all installed languages. </li></strong>
            </ul>
        <h3>Do you want to proceed ?</h3>
        """
        wizard = self.env['seo.url.key.wizard'].create(
            {'text': _(msg), 'warning_msg': True})
        return {
            'name': "Message",
            'view_mode': 'form',
            'view_id': False,
            'res_model': 'seo.url.key.wizard',
            'res_id': wizard.id,
            'type': 'ir.actions.act_window',
            'nodestroy': True,
            'target': 'new',
            'context': self._context
        }

    def unactive_all_redirects(self):
        check_list = []
        check_list.append(VAL_DICT.get(self.seo_model).replace('_', '.'))
        all_rewrites = self.env['website.rewrite'].search(
            [('rewrite_val', 'in', check_list)])
        all_rewrites.write({'active': False})

    def reset_to_draft(self):
        self.ensure_one()
        self.unactive_all_redirects()
        self.seo_cron.active = False
        self.write({
            'state': 'draft',
            'has_seo_gen': False,
            'active': False
        })

        if self.seo_model == 'product':
            self.env.cr.execute(
                "UPDATE product_template SET url_redirect_generated = FALSE")
        self.env.cr.execute(
            f"UPDATE {VAL_DICT.get(self.seo_model)} SET url_key = NULL")

    def move_to_ready(self):
        self.ensure_one()
        self.write({
            'state': 'ready',
            'active': True
        })
        if self.activate_cron:
            self.seo_cron.active = True

    def make_it_live(self):
        self.write({
            'activate_cron': False,
            'state': 'enabled',
        })

    def generate_product_seo(self):
        try:
            product_records = self.env['product.template'].search(
                [('url_redirect_generated', '=', False)])
            self.env['website.rewrite'].with_context(current_lang=self._context.get('current_lang'), new_install_lang=self._context.get('new_install_lang')).setSeoUrlKey(
                'pattern_product', product_records)
            self.has_seo_gen = True
            self.make_it_live()
        except ValidationError as ve:
            raise ValidationError(ve)
        except Exception as e:
            _logger.info(
                "@@@ Issue while creating product seo url keys  ---  %r  @@@", str(e))

    def generate_category_seo(self):
        try:
            categ_objs = self.env['product.public.category'].search([])
            self.env['website.rewrite'].with_context(current_lang=self._context.get('current_lang'), new_install_lang=self._context.get('new_install_lang')).setSeoUrlKey(
                'pattern_category', categ_objs)
            self.has_seo_gen = True
            self.make_it_live()
        except ValidationError as ve:
            raise ValidationError(ve)
        except Exception as e:
            _logger.info(
                "@@@ Issue while creating product seo url keys  ---  %r  @@@", str(e))

    @api.model
    def getCategoryUrl(self, categoryObj):
        url = '/shop/category/'
        if len(self) and categoryObj and categoryObj.url_key:
            if self.use_category_hierarchy:
                parent_url = ''
                parent_category = categoryObj.parent_id
                while parent_category:
                    parent_url = parent_category.url_key + '/' + parent_url
                    parent_category = parent_category.parent_id
                url += parent_url
                if categoryObj.url_key:
                    url += categoryObj.url_key
            else:
                url += categoryObj.url_key
            if self.use_server_rewrites:
                url = url.replace('/shop/category/', '/')
            if self.use_suffix:
                url += self.suffix_value
        else:
            url += self.env['ir.http']._slug(categoryObj)

        return url

    @api.model
    def get_seo_product_url(self, product):
        url = False
        seo_config_categ = self.search(
            [('state', '=', 'enabled'), ('seo_model', '=', 'category')], limit=1)
        if not len(self):
            product.website_url = "/shop/%s" % self.env['ir.http']._slug(
                product)

        elif seo_config_categ and seo_config_categ.use_category_hierarchy or seo_config_categ.use_category_url:
            if request:
                currentPage = request.httprequest.environ.get('PATH_INFO')
                currentPage = currentPage.replace("/shop/category/", "/")
                redirectObj = self.env['website.rewrite'].sudo(
                ).get_parent_category(currentPage)
                if redirectObj and redirectObj.get('mainCat'):
                    url = redirectObj.get('mainCat') + "/"
        url = '/shop/%s%s' % (url or '', http_slug(product))
        if self.use_server_rewrites:
            url = url.replace('/shop/', '/')
        product.website_url = url

    def _seo_bulk_product_generation(self):
        seo_config = self.env['seo.rewrite.config'].search(
            [('seo_model', '=', 'product')], limit=1)
        if seo_config.activate_cron:
            seo_config.generate_product_seo()
            # seo_config.generate_category_seo()
            seo_config.make_it_live()

    def _seo_bulk_category_generation(self):
        seo_config = self.env['seo.rewrite.config'].search(
            [('seo_model', '=', 'category')], limit=1)
        if seo_config.activate_cron:
            # seo_config.generate_product_seo()
            seo_config.generate_category_seo()
            seo_config.make_it_live()

    def seo_not_generated(self):
        action = self.env['ir.actions.act_window']._for_xml_id(
            f'seo_url_redirect.action_rewrite_{self.seo_model}')
        action['domain'] = [('url_key', '=', False)]
        return action
