# -*- coding: utf-8 -*-
#################################################################################
#
#   Copyright (c) 2015-Present Webkul Software Pvt. Ltd. (<https://webkul.com/>)
#   See LICENSE file for full copyright and licensing details.
#   If not, see <https://store.webkul.com/license.html/>
#
#################################################################################
import logging

from odoo import api, fields, models, _
from odoo.addons.seo_url_redirect.models.ir_http import slug as http_slug
from odoo.http import request
import re
from odoo.exceptions import ValidationError
from werkzeug.exceptions import NotFound
_logger = logging.getLogger(__name__)

MODEL_RULE = {'product.template': (
    'shop/', 'product'), 'product.public.category': ('shop/category/', 'category')}
PATTERN_SEO_CONFIG = {'pattern_product': 'product',
                      'pattern_category': 'category'}


class WebsiteRedirect(models.Model):
    _inherit = "website.rewrite"

    def _get_rewrites(self):
        return [('custom', 'Custom'), ('product.template', 'Product'), ('product.public.category', 'Category')]

    _rewrite_selection = lambda self, * \
        args, **kwargs: self._get_rewrites(*args, **kwargs)

    rewrite_val = fields.Selection(
        string='Create URL Rewrite',
        selection=_rewrite_selection,
        help="Name of model for rewrite management. ex : [('model.model', 'Model')]",
        default='custom'
    )
    create_type = fields.Selection(
        string='Type',
        selection=[('custom', 'Custom'), ('system', 'System')],
        help="Rewrite record create from",
        default='custom'
    )
    record_id = fields.Integer(
        string='ID Path (Model)',
        help="ID of rewrite model"
    )
    is_old_redirect = fields.Boolean('Is Old Redirect')
    last_visited_time = fields.Datetime(
        'Last Visited Time', default=lambda self: fields.Datetime.now())
    lang_id = fields.Many2one(
        "res.lang", "Language",
        default=lambda self: self.env["res.lang"].search([("code", "=", self.env.user.lang)], limit=1))

    @api.model_create_multi
    def create(self, vals):
        res = super(WebsiteRedirect, self).create(vals)
        self.env['website.route']._refresh()
        return res

    def write(self, vals):
        redirectObjs = []
        if 'url_from' in vals and 'url_to' in vals:
            redirectObjs = self.search([
                ('url_from', '=', vals.get('url_from')),
                ('url_to', '=', vals.get('url_to')),
                ('record_id', '=', self.record_id),
                ('rewrite_val', '=', self.rewrite_val),
                ('lang_id', '=', self.lang_id.id),
            ])
        elif 'url_to' in vals:
            redirectObjs = self.search([
                ('url_to', '=', vals.get('url_to')),
                ('url_from', '=', self.url_from),
                ('record_id', '=', self.record_id),
                ('rewrite_val', '=', self.rewrite_val),
                ('lang_id', '=', self.lang_id.id),
            ])
        elif 'url_from' in vals:
            redirectObjs = self.search([
                ('url_from', '=', vals.get('url_from')),
                ('url_to', '=', self.url_to),
                ('record_id', '=', self.record_id),
                ('rewrite_val', '=', self.rewrite_val),
                ('lang_id', '=', self.lang_id.id),
            ])
        if redirectObjs:
            return True
        res = super(WebsiteRedirect, self).write(vals)
        return res

    def getUrlSuffix(self, value):
        modelname = value._name
        suffix = self.getSuffix(modelname)
        return suffix

    def trackSuffix(self, modelName):
        seo_config = self.env['seo.rewrite.config'].sudo().search([], limit=1)
        use_suffix = seo_config.use_suffix
        suffix = self.env['website.rewrite'].sudo().getSuffix(modelName)
        return {'use_suffix': use_suffix, 'suffix': suffix}

    def getSuffix(self, modelname):
        seo_config = self.env['seo.rewrite.config'].sudo().search(
            [('state', '=', 'enabled'), ('seo_model', '=', MODEL_RULE.get(modelname)[1])], limit=1)
        if seo_config and seo_config.use_suffix:
            return seo_config.suffix_value
        return ''

    def unsetUrlSuffix(self, value):
        for key, val in PATTERN_SEO_CONFIG.items():
            seo_config = self.env['seo.rewrite.config'].sudo().search(
                [('state', '=', 'enabled'), ('seo_model', '=', val)], limit=1)
            if seo_config.use_suffix:
                value = value.replace(seo_config.suffix_value, '')
        return value

    def setRewriteUrl(self, data):
        url_from = "/" + data.get('url_from')
        url_to = "/" + data.get('url_to')
        redirectExistObjs = self.search([
            # ('url_from','=', url_from),
            ('record_id', '=', data.get('record_id')),
            ('rewrite_val', '=', data.get('rewrite_val')),
            ('lang_id', '=', data.get('lang_id'))
        ], limit=1)
        record_obj = self.env[data.get('rewrite_val')].browse(
            int(data.get('record_id')))
        if redirectExistObjs:
            # for redirectExistObj in redirectExistObjs:
            # if redirectExistObjs.url_to != url_to:
            redirectExistObjs.url_to = url_to
            redirectExistObjs.name = record_obj.name
            redirectObjs = redirectExistObjs
        else:
            # data['website_id'] = self.env['website'].get_current_website().id
            data['redirect_type'] = '301'
            data['create_type'] = 'system'
            data['url_from'] = url_from
            data['url_to'] = url_to
            data['name'] = record_obj.name
            redirectObjs = self.create(data)
        return redirectObjs

    def getSlugUrlKeyModel(self, value, model):
        res = self.env['product.public.category'].sudo().search(
            [('url_key', '=', value)])
        if not res:
            res = self.env['product.template'].sudo().search(
                [('url_key', '=', value)])
        if res:
            model = res._name
        return model

    def getCategoryUrl(self, categoryObj):
        catUrl = ''
        while categoryObj.parent_id:
            categoryObj = categoryObj.parent_id
            urlKey = categoryObj.url_key
            catUrl = urlKey + "/" + catUrl
        return catUrl

    def createUrlKey(self, modelObj, fieldsList):
        url_key = []
        for field in fieldsList:
            if hasattr(modelObj, field):
                name = getattr(modelObj, field)
                name = self.env['ir.http']._slugify(str(name) or '')
                url_key.append(name)
        urlKey = '-'.join(url_key)
        if not urlKey:
            urlKey = '{}-{}'.format(modelObj.id,
                                    modelObj._name.replace(".", "-"))
        return urlKey

    def getFieldList(self, pattern, model):
        website_id = self.env['website'].get_current_website()
        if hasattr(model, 'website_id'):
            website_id = model.website_id or self.env['website'].get_current_website(
            )
        fieldsList = []
        seo_config = self.env['seo.rewrite.config']
        if model._name:
            seo_config = self.env['seo.rewrite.config'].sudo().search(
                [('seo_model', '=', MODEL_RULE.get(model._name, ['', ''])[1])], limit=1)
        else:
            seo_config = self.env['seo.rewrite.config'].sudo().search(
                [('seo_model', '=', PATTERN_SEO_CONFIG.get(pattern))], limit=1)
        if seo_config:
            modelPattern = seo_config.mapped('pattern_url_key')[0]
            if modelPattern:
                fieldsList = modelPattern.split(',')
        return fieldsList

    def get_old_url(self, value):
        identifier, name = value.id, getattr(
            value, 'seo_name', False) or value.display_name
        slugname = self.env['ir.http']._slugify(name or '').strip().strip('-')
        return "%s-%d" % (slugname, identifier)

    def get_installed_langs(self):
        return self.env['res.lang'].with_context(active_test=True).search([])

    def createRedirectForRewrite(self, vals, modelObj, lang):
        modelName = modelObj._name
        pattern = 'pattern_product' if modelName == 'product.template' else 'pattern_category'
        if 'url_key' in vals:
            vals['url_key'] = re.sub(
                '[^A-Za-z0-9]+[//]*[^A-Za-z0-9]+[//]*', '-', vals['url_key'])
            # oldUrl = modelObj.url_key
            # if oldUrl in ['', False, None]:
            oldUrl = self.get_old_url(modelObj)
            urlTo = vals.get('url_key')
            if urlTo in ['', False, None]:
                fieldsList = self.getFieldList(pattern, modelObj)
                urlKey = self.createUrlKey(modelObj, fieldsList)
                urlTo = urlKey
                vals['url_key'] = urlTo
            old_url_route_prefix = MODEL_RULE.get(modelName, ['', ''])[0]
            redirectData = {
                'rewrite_val': modelName,
                'record_id': modelObj.id,
                'url_from': old_url_route_prefix+oldUrl,
                'url_to': urlTo,
                'lang_id': lang.id
            }
            self.setRewriteUrl(redirectData)
        return vals

    def _duplicate_last_redirect(self, modelObj, lang, latest_objs, old_objs, vals):
        # new_url_key = "/" + modelObj.with_context(lang=lang).url_key
        new_url_key = "/" + vals.get('url_key')
        latest_objs.write({
            'name': latest_objs.url_to,
            'url_from': latest_objs.url_to,
            'url_to': new_url_key

        })
        old_obj_ids = old_objs.mapped('id')
        if len(old_obj_ids):
            id_tupl = tuple(old_obj_ids) if len(
                old_obj_ids) > 1 else "('{}')".format(old_obj_ids.pop())
            url_to_vals = {
                'ids': id_tupl,
                'url_to': new_url_key,
                'is_old_redirect': True,
            }
            self._update_url_to(url_to_vals)

    def _update_url_to(self, vals):
        self._cr.execute(
            f'''UPDATE website_rewrite set url_to = '{vals.get("url_to")}',is_old_redirect = {vals.get('is_old_redirect')}   where id in {vals.get('ids')}''')
        self._cr.commit()

    def updateSeoUrlKey(self, modelObjs, lang='en_US', vals={}):
        failedIds = []
        for modelObj in modelObjs:
            try:
                current_lang = self.env['res.lang'].search(
                    [('code', '=', lang)], limit=1)
                if current_lang:
                    rewrite_objs = self.search([('rewrite_val', '=', modelObj._name), (
                        'lang_id', '=', current_lang.id), ('record_id', '=', modelObj.id)])
                    modelObj = modelObj.with_context(lang=lang)
                    if not rewrite_objs and vals.get('url_key'):
                        # if not vals.get('url_key'):
                        return self.env['website.rewrite'].sudo().createRedirectForRewrite(vals, modelObj, current_lang)

                    if not vals.get('url_key'):
                        pattern = vals.get('pattern')
                        url_key = self.getUrlKey(lang, pattern, modelObj)
                        vals.update({'url_key': url_key})

                    old_objs = rewrite_objs[0] if rewrite_objs else rewrite_objs
                    active_rewrite_obj = rewrite_objs.filtered(
                        lambda obj: not obj.is_old_redirect)
                    active_rewrite_obj = active_rewrite_obj[0] if active_rewrite_obj else active_rewrite_obj
                    if active_rewrite_obj and active_rewrite_obj.url_to[1::] != vals.get('url_key'):
                        latest_objs = active_rewrite_obj.copy()
                        self._duplicate_last_redirect(
                            modelObj, lang, latest_objs, old_objs, vals)
                    vals.update({'url_key': ''})
            except ValidationError as e:
                if modelObj._name == 'product.template':
                    failedIds.append(modelObj.default_code or '')
            finally:
                pass
        return failedIds

    def getUrlKey(self, lang, pattern, modelObj):
        if modelObj:
            fieldsList = self.getFieldList(pattern, modelObj)
            modelObj = modelObj.with_context(lang=lang)
            urlKey = self.createUrlKey(modelObj, fieldsList)
            turlKey = False
            try:
                # Updating urlkey in given language
                if lang:
                    modelObj = modelObj.with_context(lang=lang)
                    turlKey = self.createUrlKey(modelObj, fieldsList)
                    modelObj.url_key = turlKey if turlKey else urlKey
            except ValidationError as e:
                _logger.info(
                    f"###### Exception : {str(e)} while updating the url keys of : {modelObj}############")
        return turlKey if turlKey else urlKey

    def setSeoUrlKey(self, pattern, modelObjs):
        failedIds = []
        # Getting all active languages
        langs = [i for i, j in self.env['res.lang'].get_installed()]
        if self._context.get('new_install_lang'):
            langs = [self._context.get('current_lang')]
        count = 0
        for modelObj in modelObjs:
            count = count+1
            fieldsList = self.getFieldList(pattern, modelObj)
            urlKey = self.createUrlKey(modelObj, fieldsList)
            try:
                # Updating urlkey in all languages
                for lang in langs:
                    modelObj = modelObj.with_context(lang=lang)
                    turlKey = self.createUrlKey(modelObj, fieldsList)
                    url_key = turlKey if turlKey else urlKey
                    if modelObj.url_key != url_key:
                        modelObj.with_context(lang=lang).write(
                            {'url_key': url_key})
                    else:
                        redirects = self.search(
                            [('url_to', '=', url_key), ('lang_id.code', '=', lang)])
                        if not redirects:
                            self.updateSeoUrlKey(
                                modelObj, lang, {'url_key': url_key})
                if modelObj._name == 'product.template' and not modelObj.url_redirect_generated:
                    modelObj.write({'url_redirect_generated': True})
            except ValidationError as e:
                _logger.info(
                    f"###### Exception : {str(e)} while updating the url keys of : {modelObj}############")
            finally:
                if count == 100:
                    self.env.cr.commit()
                    count = 0

        return failedIds

    def get_parent_category(self, hrefUrl=''):
        frontEndLang = request.httprequest.cookies.get('frontend_lang')
        seo_config_pro = self.env['seo.rewrite.config'].sudo().search(
            [('state', '=', 'enabled'), ('seo_model', '=', 'product')], limit=1)
        # website_id = request.env['website'].get_current_website()
        seo_config_categ = self.env['seo.rewrite.config'].sudo().search(
            [('state', '=', 'enabled'), ('seo_model', '=', 'category')], limit=1)
        useCategoryUrl = seo_config_categ.use_category_url
        useServerRewrites = seo_config_categ.use_server_rewrites
        if not useCategoryUrl:
            return {"mainCat": False, "use_server_rewrites": useServerRewrites}
        if not seo_config_categ and useCategoryUrl:
            arr_url = hrefUrl.split('/')
            category_url = arr_url[-1] if arr_url else None
            if category_url:
                cat_id_arr = category_url.split('-')
                cat_id = cat_id_arr[-1] if cat_id_arr else None
                if cat_id and cat_id.isdigit():
                    categoryObj = self.env['product.public.category'].sudo().browse(
                        int(cat_id))
                    if categoryObj.exists():
                        return {"mainCat": hrefUrl[1::], "use_server_rewrites": seo_config_pro.use_server_rewrites}

        redirectObj = self.env['website.rewrite'].sudo()
        categoryUrlKey = redirectObj.unsetUrlSuffix(hrefUrl)
        categoryUrlKey = ''.join(categoryUrlKey.split('/')[-1:])
        categoryObj = self.env['product.public.category'].sudo().search(
            [('url_key', '=', categoryUrlKey)], limit=1)
        catUrl = ''
        if categoryObj:
            catUrl = categoryObj.url_key
            if seo_config_categ.use_category_hierarchy:
                while categoryObj.parent_id:
                    categoryObj = categoryObj.parent_id
                    urlKey = categoryObj.url_key
                    catUrl = urlKey + "/" + catUrl
        if catUrl == '':
            catUrl = False
        return {"mainCat": catUrl, "use_server_rewrites": useServerRewrites, 'lang': frontEndLang}

    def linked_product(self):
        model = self.rewrite_val
        record_id = self.record_id
        action = False
        if model == 'product.template':
            action = self.env['ir.actions.act_window']._for_xml_id(
                'seo_url_redirect.action_rewrite_product')
        elif model == 'product.public.category':
            action = self.env['ir.actions.act_window']._for_xml_id(
                'seo_url_redirect.action_rewrite_category')
        if action:
            action['domain'] = [('id', '=', record_id)]
            return action
        return False


class Website(models.Model):
    _inherit = "website"

    def get_seo_category_url(self, categ):
        self.ensure_one()
        seo_config = self.env['seo.rewrite.config'].sudo().search(
            [('state', '=', 'enabled'), ('seo_model', '=', 'category')], limit=1)
        return seo_config.getCategoryUrl(categ)

    def _enumerate_pages(self, query_string=None, force=False):
        res = super(Website, self)._enumerate_pages(query_string, force)
        useServerRewrites_pro = self.env['seo.rewrite.config'].sudo().search(
            [('state', '=', 'enabled'), ('seo_model', '=', 'product')], limit=1).use_server_rewrites
        useServerRewrites_categ = self.env['seo.rewrite.config'].sudo().search(
            [('state', '=', 'enabled'), ('seo_model', '=', 'category')], limit=1).use_server_rewrites
        for rec in res:
            if 'loc' in rec:
                if '/shop/category' in rec['loc'] and useServerRewrites_categ:
                    rec['loc'] = rec['loc'].replace(
                        "/shop/category/", "/").replace('__', '/')
                elif '/shop/' in rec['loc'] and useServerRewrites_pro:
                    rec['loc'] = rec['loc'].replace(
                        "/shop/", "/").replace('__', '/')
                yield rec
